/*********************************************************************************
   High speed driver for 122x32 SG12232A graphics LCD from altronics.com.au
   Using the SED1520 chipset

   Ver 1.0 - Apr 09 - Original version
   Ver 2.0 - Jul 09 - High speed version

   Go to http://geoffg.net/GPS_Display.html for updates, errata and helpfull notes
    
   Copyright 2009 Geoff Graham - http://geoffg.net
   This program is free software: you can redistribute it and/or modify it under the terms of the GNU General
   Public License as published by the Free Software Foundation, either version 2 of the License, or (at your
   option) any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
   implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
   for more details.  You should have received a copy of the GNU General Public License along with this program.
   If not, see <http://www.gnu.org/licenses/>.
   
   This is a high speed version of the original driver.  It uses a frame buffer to build
   the graphic image, the data is then copied at high speed to the LCD.  It will only
   work on chips such as the 18F series with sufficient RAM as the frame buffer uses 
   488 bytes.  It is modified to suit the GPS Car Computer project published in Silicon Chip
   which mounts the display upside down and has the data bus inverted.
   
   Target/Development environment:
    - Microchip PIC microcontroller chips
    - Microchip MPLAB V8 or later
    - CCS C Compiler V4 or later (tested)
    - HiTech C Compiler (not tested)
    
************************************************************************************
    
   This driver exposes the following functions and defines.

   glcd_init()
      * Initialises the display
      * Must be called before any other function.

   glcd_pixel(x,y,colour)
      * Sets the pixel to the given colour.
        - origin (x=0, y=0) is top left corner
        - colour can be ON or OFF (0 or 1)
        - note that this does not update the display

   glcd_fillScreen(colour)
      * Fills the entire LCD with the given colour.
        - colour can be ON or OFF (0 or 1)
        - note that this does not update the display

   glcd_update()
      * Updates the LCD by copying the frame buffer to the LCD
        - on a 48MHz chip this takes 6.5mS to execute

   GLCD_GRAPHICS_WIDTH = 122     Width of display
   GLCD_GRAPHICS_HEIGHT = 32     Height of display

************************************************************************************

    The following pins must be defined
        LCD_CS2      output     LCD Chip Select 1
        LCD_CS1      output     LCD Chip Select 2
        LCD_A0       output     LCD A0
        LCD_E        output     LCD Enable
        LCD_RW       output     LCD Read/Write - this is optional as we never read
                                from the chip.  In the circuit this pin could be
                                wired to ground.
    
    These should be defined as a writable bit corresponding to the pin on the chip
    For example, in CCS C:
        #byte   porta = 5                   // Port A I/O register
        #bit    LCD_CS2 = porta.0           // RA0 (i.e.  Port A bit 0)
    Or, in HiTech C:
        #define LCD_CS2   RA0
    
    The 8 bit port used for the data bus must be defined as LCD_DATA and its associated
    tris register set to output (we never read from the LCD).
    
    For example, in CCS C:
        #byte LCD_DATA = 7          // define the port c register
        #byte TRISC = 0x87          // define the port c tris register
        TRISC = 0;                  // make output

    In addition, the control pins (LCD_CS2, etc) must set to outputs before calling 
    glcd_init()

    Finally, in CCS C the ports used should be set to fast I/O.  For example:
        #use fast_io(A) 
        #use fast_io(C)


************************************************************************************/

// Display commands
#define DISPLAY_ON 0xAF
#define PAGE_ADDRESS_SET 0xB8
#define COLUMN_ADDRESS_SET 0x00

#define SCREEN_WIDTH    122
#define SCREEN_HEIGHT 32

#define TRIS_OUTPUT         0
#define TRIS_INPUT          0xff

uint8 Fb[SCREEN_WIDTH][SCREEN_HEIGHT/8];                    // frame buffer for building the LCD image


//-------------------------------------------------------------------------------------------------
// Turn on or off all pixels
//-------------------------------------------------------------------------------------------------
void glcd_fillscreen(uint8 colour) {
    char x, y;
    
    if(colour) colour = 0xff;       
    for(x = 0; x < SCREEN_WIDTH; x++)   
        for(y = 0; y < SCREEN_HEIGHT/8; y++)
            Fb[x][y] = colour;
}


//-------------------------------------------------------------------------------------------------
// Write a pixel, either off or on
//-------------------------------------------------------------------------------------------------
void glcd_pixel(uint8 x, uint8 y, uint8 colour) {
    if(x >= SCREEN_WIDTH || y >= SCREEN_HEIGHT) return;
    if(colour)
        Fb[x][y >> 3] |= (1 << (y & 0b00000111));
    else
        Fb[x][y >> 3] &= ~(1 << (y & 0b00000111));
}



//-------------------------------------------------------------------------------------------------
// Write a command
// The data bus is inverted because the LCD is installed upside down.  Inversion suits the data (because
// the image must be inverted) but messes up any commands written to the LCD.  So, before the command
// is sent it is inverted - which cancels the inversion on the PCB.
//-------------------------------------------------------------------------------------------------
void glcd_WriteCommand(uint8 i) {
    #bit jb0 = LCD_DATA.0
    #bit jb1 = LCD_DATA.1
    #bit jb2 = LCD_DATA.2
    #bit jb3 = LCD_DATA.3
    #bit jb4 = LCD_DATA.4
    #bit jb5 = LCD_DATA.5
    #bit jb6 = LCD_DATA.6
    #bit jb7 = LCD_DATA.7

    LCD_A0 = 0;
    jb0 = bit_test(i, 7);                                   // write the bits, inverting as we go
    jb1 = bit_test(i, 6);
    jb2 = bit_test(i, 5);
    jb3 = bit_test(i, 4);
    jb4 = bit_test(i, 3);
    jb5 = bit_test(i, 2);
    jb6 = bit_test(i, 1);
    jb7 = bit_test(i, 0);
    LCD_E = 1;
    LCD_A0 = 0;                                             // this is essentially a nop to provide an access delay
    LCD_E = 0;
    LCD_A0 = 1; 
}



//-------------------------------------------------------------------------------------------------
// Initialise the display
// Only needs to be called once at the start of the program
//-------------------------------------------------------------------------------------------------
void glcd_init(void) {
    #ifdef LCD_RW
        LCD_RW = 0;                                         // we never read from the display so this is optional
    #endif
    #ifdef LCD_DATA_TRIS
        LCD_DATA_TRIS = TRIS_OUTPUT;                        // same for the data direction
    #endif
    LCD_CS1 = 0; LCD_CS2 = 1; glcd_WriteCommand(DISPLAY_ON);
    LCD_CS1 = 1; LCD_CS2 = 0; glcd_WriteCommand(DISPLAY_ON);
    LCD_CS2 = 1; 
    glcd_fillscreen(0);
}



//-------------------------------------------------------------------------------------------------
// Copy the frame buffer to the LCD
//-------------------------------------------------------------------------------------------------
void glcd_update(void) {
    uint8 x, y;

    for(y = 0; y < SCREEN_HEIGHT/8; y++) {
        LCD_CS1 = 0;
        glcd_WriteCommand(COLUMN_ADDRESS_SET);
        glcd_WriteCommand(PAGE_ADDRESS_SET | y);
        for(x = 0; x < SCREEN_WIDTH/2; x++) {
            LCD_DATA = Fb[(SCREEN_WIDTH - 1) - x][((SCREEN_HEIGHT/8) - 1) - y]; 
            LCD_E = 1;                              // strobe the data
            LCD_A0 = 1;                             // this is essentially a nop to provide an access delay
            LCD_E = 0;
        }   
        LCD_CS1 = 1; LCD_CS2 = 0;
        glcd_WriteCommand(COLUMN_ADDRESS_SET);
        glcd_WriteCommand(PAGE_ADDRESS_SET | y);
        for(; x < SCREEN_WIDTH; x++) {
            LCD_DATA = Fb[(SCREEN_WIDTH - 1) - x][((SCREEN_HEIGHT/8) - 1) - y]; 
            LCD_E = 1;                              // strobe the data
            LCD_A0 = 1;                             // this is essentially a nop to provide an access delay
            LCD_E = 0;
        }
        LCD_CS2 = 1;
    }   
}
